<?php

namespace App\Console\Commands;

use App\Services\FetchNotikOffers;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Http;
use App\Models\Offer;
use App\Models\Setting;
use Illuminate\Support\Facades\Cache;

class FetchOffers extends Command
{
    protected $signature = 'FetchOffers:cron';

    protected $description = 'Fetch offers from Adgate Media';

    private $settings;

    public function __construct()
    {
        parent::__construct();
    }

    public function handle()
    {
        $this->loadSettings();
        if (isAdgateApiEnabled()) {
            $this->fetchAdgateOffers();
        }

        if (isNotikApiEnabled()) {
            (new FetchNotikOffers())->get();
        }


        $this->info('Offers fetched and stored successfully!');
    }

    private function loadSettings()
    {
        $this->settings = Cache::remember('apiSettings', 3600, function () {
            return Setting::whereIn('name', [
                'adgate_affiliate_id',
                'adgate_api_key',
                'adgate_wall',
                'adgate_rate',
            ])->pluck('value', 'name')->toArray();
        });
    }
 
    private function fetchAdgateOffers()
    {
        try {
            $url = 'https://api.adgatemedia.com/v3/offers?' . http_build_query([
                'aff' => $this->settings['adgate_affiliate_id'],
                'api_key' => $this->settings['adgate_api_key'],
                'wall_code' => $this->settings['adgate_wall'],
                'categories' => '21,22,23',
                'minepc' => 1,
            ]);

            $response = Http::get($url);

            if (!$response->successful()) {
                \Log::error('Adgate API Request failed.', ['status' => $response->status()]);
                return;
            }

            $offers = $response->json('data', []);

            foreach ($offers as $offer) {
                $this->processAdgateOffer($offer);
            }
        } catch (\Exception $e) {
            \Log::error('Error fetching Adgate offers: ' . $e->getMessage());
        }
    }

    private function processAdgateOffer($offer)
    {
        $countries = array_column($offer['geo_targeting']['countries'] ?? [], 'country_code');
        $operatingSystems = array_column($offer['device_targeting']['operating_systems'] ?? [], 'name');

        $events = array_map(function ($event) {
            return [
                'id' => $event['id'],
                'name' => $event['name'],
                'payout' => $event['payout'] * $this->settings['adgate_rate']
            ];
        }, $offer['events'] ?? []);

        $totalPayout = array_sum(array_column($events, 'payout'));
        $adjustedPayout = $totalPayout;

        Offer::updateOrCreate(
            ['offer_id' => $offer['id']],
            [
                'name' => htmlspecialchars_decode($offer['anchor'] ?? '', ENT_QUOTES),
                'description' => htmlspecialchars_decode($offer['description'] ?? '', ENT_QUOTES),
                'requirements' => $offer['requirements'] ?? '',
                'payout' => $adjustedPayout,
                'creative' => $offer['creatives']['icon'] ?? '',
                'event' => json_encode($events),
                'device' => implode(', ', $operatingSystems),
                'partner' => 'adgatemedia',
                'link' => $offer['click_url'] ?? '',
                'countries' => implode(',', $countries),
            ]
        );
    }

}
