<?php

namespace App\Services;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Http;
use App\Models\Setting;
use App\Models\Offer;
use App\Models\Track;

class OfferService
{
    protected $partners = ['ogads', 'adgatemedia', 'notik'];

    protected $settings;

    public function getOffers($partner, $userCountry, $numoffers = null)
    {

        if($partner == 'adgatemedia' && !isAdgateApiEnabled()) return [];

        if($partner == 'notik' && !isNotikApiEnabled()) return [];
 
        $user = Auth::user();
        $completedOfferIds = $user ? $this->getCompletedOfferIds($user) : [];

        $deviceType = detectDevicePlatform();

        $query = Offer::where('partner', $partner)
            ->whereRaw('FIND_IN_SET(?, countries)', [$userCountry])
            ->when($deviceType !== '', function ($q) use ($deviceType) {
                $q->where(function ($query) use ($deviceType) {
                    $query->where('device', $deviceType)
                        ->orWhere('device', '')
                        ->orWhere('device', 'all');
                });
            })
            ->orderByDesc('payout');

        if ($numoffers) {
            $query->limit($numoffers);
        }

        return $this->transformOffers($query->get(), $completedOfferIds);
    }

    protected function transformOffers($offers, $completedOfferIds)
    {
        return $offers->reject(fn($offer) => in_array($offer->id, $completedOfferIds))
            ->map(function ($offer) {
                $events = is_string($offer->event) ? json_decode($offer->event, true) : $offer->event;
                $events = is_array($events) ? $events : [];
                $offer->event = array_map(fn($event) => [
                    'name' => $event['name'] ?? null,
                    'payout' => $event['payout'] ?? null,
                ], $events);
                return $offer;
            })->values();
    }




    public function fetchOgadsOffers(Request $request)
    {

        if (isOgadsApiEnabled()) {
            $user = Auth::user();
            $completedOfferIds = [];

            // Only fetch completed offers if user is authenticated
            if ($user) {
                $completedOfferIds = $this->getCompletedOfferIds($user);
            }

            $settings = Setting::whereIn('name', ['ogads_api_key', 'ogads_rate'])->pluck('value', 'name');
            $apiKey = $settings['ogads_api_key'] ?? null;
            $ogRate = $settings['ogads_rate'] ?? 1;

            if (!$apiKey) {
                throw new \Exception('Ogads API key not found in settings.');
            }

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->get('https://unlockcontent.net/api/v2', [
                'ip' => $request->ip(),
                'user_agent' => $request->header('User-Agent'),
                'ctype' => 3,
            ]);

            if ($response->failed()) {
                throw new \Exception('Failed to fetch Ogads offers: ' . $response->status());
            }

            $offers = $response->json('offers', []);


            return collect(array_filter($offers, fn($offer) => !$user || !in_array($offer['offerid'], $completedOfferIds))) // Skip filtering if user is not authenticated
                ->map(fn($offer) => ['payout' => $offer['payout'] * $ogRate] + $offer)
                ->values();
        }

        return collect([]);
    }


    public function fetchNotikOffers(Request $request)
    {

        if (isNotikApiEnabled()) {

            $user = Auth::user();
            $completedOfferIds = [];

            // Only fetch completed offers if user is authenticated
            if ($user) {
                $completedOfferIds = $this->getCompletedOfferIds($user);
            }

            $deviceType = detectDevicePlatform();

            $query = Offer::where('partner', $partner)
                ->whereRaw('FIND_IN_SET(?, countries)', [$userCountry])
                ->when($deviceType !== '', function ($q) use ($deviceType) {
                    $q->where(function ($query) use ($deviceType) {
                        $query->where('device', $deviceType)
                            ->orWhere('device', '');
                    });
                })
                ->orderByDesc('payout');

            if ($numoffers) {
                $query->limit($numoffers);
            }

            return $this->transformOffers($query->get(), $completedOfferIds);
        }

        return collect([]);
    }

    protected function getCompletedOfferIds($user)
    {
        return Track::where('uid', $user->uid)
            ->whereIn('partners', $this->partners)
            ->pluck('offer_id')
            ->toArray();
    }
}
